//-------------------------------------------------------------------------
//			ActionMedia II Programmer's Toolkit
//			
//			Windows Sample Code Shared Functions
//
// Module Name:	log.c
//
// Description:	This module contains functions for creating a log file
//				and writing log messages to it.
//
//	Copyright Intel Corp. 1991, 1992
//	All Rights Reserved.
//
//-------------------------------------------------------------------------
//
// Exported functions from this module:
//
//		LogOpen
//		LogWrite
//		LogMsg
//	    LogClose
//		LogOn
//		LogOff
//		LogStat
//
//-------------------------------------------------------------------------
// NOTES on Log
//	
//	These functions are used in most of the sample applications to 
//	manage a log file for monitoring an application's performance and
//	for debugging.
//
//-------------------------------------------------------------------------
#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <io.h>
#include <fcntl.h>
#include <errno.h>
#define	LOG_NOEXTERNS
#include "log.h"

static int		hFile = -1;
static int		fLogFlag = FALSE;
static char		Buf[256];

//-------------------------------------------------------------------------
//FUNCTION:
//
//	BOOL	LogOpen(pFileSpec)
//
//PARMS IN:
//	
//	char *pFileSpec		pointer to string with file name to use in
//							creating the log file. 
//	
//DESCRIPTION:
//	
//	This function creates a log file with the passed in name for use by 
//	the logging subsystem. If a file already exists with the name, it
//	will be truncated. 
//
//	There may only be one log file open at a time.
//	
//RETURN:
//	
//	TRUE on success.
//	FALSE on failure (no error message is displayed to the user).
//	
//-------------------------------------------------------------------------

BOOL 
LogOpen(char *pFileSpec)
{
	OFSTRUCT	Of;

	//	If the log file is not opened, open it now.

	if (hFile == -1)
	{
		if ((hFile = OpenFile((LPSTR)pFileSpec, (LPOFSTRUCT)&Of, 
		  OF_CREATE | OF_WRITE | OF_SHARE_EXCLUSIVE)) == -1)
			return FALSE;
	}
	fLogFlag = TRUE;
	return TRUE;
}

//-------------------------------------------------------------------------
//FUNCTION:
//
//	BOOL	LogWrite(char *pFormat, ...)
//
//PARMS IN:
//	
//	char *pFormat		Format string similar to that used by printf().
//	
//DESCRIPTION:
//	
//	This function writes out a message to the log file.  It works like
//	printf() taking a format string and a variable number of values
//	to be formatted. It does NOT append a newline. This is left to 
//	the application. The manifest constant CR, which is #defined in  
//	log.h, can be concatenated as follows:
//
//		LogWrite("Error code = %d"CR, ErrCode);
//
//	Note that ANSI string concatenation is used to add CR to the 
//	format string rather than using a separate format argument.
//	
//RETURN:
//	
//	TRUE on success.
//	FALSE on failure (no error message is displayed to the user).
//	
//-------------------------------------------------------------------------

BOOL 
LogWrite(char *pFormat, ...)
{
	va_list	pArgs;
	int		hDup;

	//	Logging off, just leave without an error.

	if (!fLogFlag)
		return TRUE;

	//	Logging on, but log file not opened. This is an error.

	if (hFile == -1)
		return FALSE;

	//	Set up to process the variable number of arguments.

	va_start(pArgs, pFormat);

	//	Format the arguments and string.

	vsprintf(Buf, pFormat, pArgs);
	
	//	Write out the formatted string.

	_lwrite(hFile, (LPSTR)Buf, strlen(Buf));

	//	Get a duplicate handle to the log file and close it.  This is
	//	a way to force a flush of the internal file buffer to the disk
	//	so a subsequent crash will not lose the message.

	if ((hDup = dup(hFile)) != -1)
		_lclose(hDup);

	//	Clean up the variable argument processing.

	va_end(pArgs);

	return TRUE;
}

//-------------------------------------------------------------------------
//FUNCTION:
//
//	BOOL	LogMsg(pMsg)
//
//PARMS IN:
//	
//	char *pMsg			pointer to the message string to be written 
//							to the log file.
//	
//DESCRIPTION:
//	
//	This function writes out a message to the log file. It is a simple
//	function that can be used when the application doesn't need to 
//	include data from variables.  This function DOES append a newline
//	to the caller's string.
//
//RETURN:
//	
//	TRUE on success.
//	FALSE on failure (no error message is displayed to the user).
//	
//-------------------------------------------------------------------------

BOOL 
LogMsg(char *pMsg)
{
	int		hDup;

	//	Logging off, just leave without an error.
	
	if (!fLogFlag)
		return TRUE;

	//	Logging on, but log file not opened. This is an error.

	if (hFile == -1)
		return FALSE;

	//	Concatenate the newline.

	sprintf(Buf, "%s"CR, pMsg);

	//	Write it to the log file.

	_lwrite(hFile, (LPSTR)Buf, strlen(Buf));

	//	Flush the file as we did in LogWrite().

	if ((hDup = dup(hFile)) != -1)
		close(hDup);

	return TRUE;
}

//-------------------------------------------------------------------------
//FUNCTION:
//
//	BOOL LogClose(VOID)
//
//DESCRIPTION:
//	
//	Closes the log file.
//	
//RETURN:
//	
//	TRUE on success.
//	FALSE on failure (no error message is displayed to the user).
//	
//-------------------------------------------------------------------------

BOOL 
LogClose()
{
	if (hFile != -1)
	{
		if (_lclose(hFile))
			return FALSE;
		hFile = -1;
	}

	fLogFlag = FALSE;
	return TRUE;
}

//-------------------------------------------------------------------------
//FUNCTION:
//
//	VOID LogOn(VOID)
//
//DESCRIPTION:
//	
//	Turns logging on. Can be used in conjunction with LogOff() to turn
//	loggin on and off within an application.
//	
//-------------------------------------------------------------------------

VOID 
LogOn()
{
	if (hFile != -1)
		fLogFlag = TRUE;
}

//-------------------------------------------------------------------------
//FUNCTION:
//
//	VOID LogOff(VOID)
//
//DESCRIPTION:
//	
//	Turns logging off.
//	
//-------------------------------------------------------------------------

VOID 
LogOff()
{
	if (hFile != -1)
		fLogFlag = FALSE;
}

//-------------------------------------------------------------------------
//FUNCTION:
//
//	BOOL	LogStat(VOID)
//
//DESCRIPTION:
//	
//	Returns the status of logging (on or off).
//	
//RETURN:
//	
//	TRUE if logging is on.
//	FALSE if logging is off.
//	
//-------------------------------------------------------------------------

BOOL 
LogStat()
{
	return fLogFlag;
}


